<?php
/**
 * Core plugin class.
 *
 * Boots dependencies, registers hooks, and initialises components.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale {

	/** @var GotASale|null */
	private static $instance = null;

	/**
	 * Returns the singleton instance.
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor — load dependencies and register hooks.
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->register_hooks();
	}

	/**
	 * Load required class files.
	 */
	private function load_dependencies() {
		require_once GOTASALE_PLUGIN_DIR . 'includes/class-gotasale-api.php';
		require_once GOTASALE_PLUGIN_DIR . 'includes/class-gotasale-woocommerce.php';
		require_once GOTASALE_PLUGIN_DIR . 'admin/class-gotasale-admin.php';
		require_once GOTASALE_PLUGIN_DIR . 'admin/class-gotasale-settings.php';
	}

	/**
	 * Register WordPress hooks.
	 */
	private function register_hooks() {
		// WooCommerce dependency check
		add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );

		// Registration retry cron
		add_action( 'gotasale_retry_registration', array( 'GotASale_Activator', 'retry_registration' ) );

		// Backward compat: if token exists but registration status is missing, schedule a check
		if ( get_option( 'gotasale_site_token' ) && ! get_option( 'gotasale_registration_status' ) ) {
			update_option( 'gotasale_registration_status', array( 'status' => 'unknown' ) );
			if ( ! wp_next_scheduled( 'gotasale_retry_registration' ) ) {
				wp_schedule_single_event( time() + 5, 'gotasale_retry_registration' );
			}
		}

		// Admin
		if ( is_admin() ) {
			$admin = new GotASale_Admin();
			$admin->init();

			$settings = new GotASale_Settings();
			$settings->init();
		}

		// WooCommerce hooks
		$woocommerce = new GotASale_WooCommerce();
		$woocommerce->init();
	}

	/**
	 * Show admin notice if WooCommerce is not active.
	 */
	public function woocommerce_missing_notice() {
		if ( class_exists( 'WooCommerce' ) ) {
			return;
		}

		printf(
			'<div class="notice notice-error"><p>%s</p></div>',
			esc_html__( 'Got A Sale requires WooCommerce to be installed and active. Please install and activate WooCommerce to use Got A Sale.', 'got-a-sale' )
		);
	}
}
