<?php
/**
 * Plugin activation handler.
 *
 * Generates a unique site token and stores default settings.
 * Handles registration with the GotASale API server with retry support.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_Activator {

	/**
	 * Backoff delays in seconds for retry attempts.
	 * 1m → 5m → 30m → 2h → 6h → 12h → 24h
	 */
	private static $backoff_delays = array( 60, 300, 1800, 7200, 21600, 43200, 86400 );

	/**
	 * Maximum number of registration attempts.
	 */
	private static $max_attempts = 10;

	/**
	 * Run on plugin activation.
	 */
	public static function activate() {
		// Generate site token if not already set
		if ( ! get_option( 'gotasale_site_token' ) ) {
			$token = wp_generate_uuid4();
			update_option( 'gotasale_site_token', $token );
		}

		// Register store with GotASale server
		self::register_store();
	}

	/**
	 * Register this store with the GotASale API server.
	 */
	private static function register_store() {
		$token    = get_option( 'gotasale_site_token' );
		$site_url = get_site_url();
		$name     = get_bloginfo( 'name' );

		if ( ! $token ) {
			return;
		}

		$response = wp_remote_post( GOTASALE_API_URL . '/stores/register', array(
			'headers'   => array(
				'Content-Type' => 'application/json',
			),
			'body'      => wp_json_encode( array(
				'site_token' => $token,
				'store_url'  => $site_url,
				'store_name' => $name,
			) ),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		if ( is_wp_error( $response ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: Failed to register store — ' . $response->get_error_message() );
			}
			self::save_registration_failure( $response->get_error_message(), 1 );
			self::schedule_retry( 1 );
			return;
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 === $code && ! empty( $body['ok'] ) ) {
			update_option( 'gotasale_registration_status', array(
				'status'       => 'registered',
				'registered_at' => time(),
			) );

			// Trigger WC API key setup after successful registration
			if ( class_exists( 'GotASale_WC_API' ) ) {
				GotASale_WC_API::setup();
			}
		} else {
			$error = isset( $body['error'] ) ? $body['error'] : 'HTTP ' . $code;
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: Registration failed — ' . $error );
			}
			self::save_registration_failure( $error, 1 );
			self::schedule_retry( 1 );
		}
	}

	/**
	 * Retry registration with exponential backoff.
	 * Called by WP Cron or manually via AJAX.
	 *
	 * @return array Status array with result of the attempt.
	 */
	public static function retry_registration() {
		$status = get_option( 'gotasale_registration_status', array() );

		// Already registered — nothing to do
		if ( isset( $status['status'] ) && 'registered' === $status['status'] ) {
			return $status;
		}

		$attempts = isset( $status['attempts'] ) ? (int) $status['attempts'] : 0;

		// Exceeded max attempts
		if ( $attempts >= self::$max_attempts ) {
			return $status;
		}

		$token    = get_option( 'gotasale_site_token' );
		$site_url = get_site_url();
		$name     = get_bloginfo( 'name' );

		if ( ! $token ) {
			return $status;
		}

		$new_attempt = $attempts + 1;

		$response = wp_remote_post( GOTASALE_API_URL . '/stores/register', array(
			'headers'   => array(
				'Content-Type' => 'application/json',
			),
			'body'      => wp_json_encode( array(
				'site_token' => $token,
				'store_url'  => $site_url,
				'store_name' => $name,
			) ),
			'timeout'   => 15,
			'sslverify' => true,
		) );

		if ( is_wp_error( $response ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'GotASale: Retry #' . $new_attempt . ' failed — ' . $response->get_error_message() );
			}
			self::save_registration_failure( $response->get_error_message(), $new_attempt );
			self::schedule_retry( $new_attempt );
			return get_option( 'gotasale_registration_status' );
		}

		$code = wp_remote_retrieve_response_code( $response );
		$body = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 === $code && ! empty( $body['ok'] ) ) {
			$new_status = array(
				'status'        => 'registered',
				'registered_at' => time(),
			);
			update_option( 'gotasale_registration_status', $new_status );

			// Trigger WC API key setup after successful registration
			if ( class_exists( 'GotASale_WC_API' ) ) {
				GotASale_WC_API::setup();
			}

			return $new_status;
		}

		$error = isset( $body['error'] ) ? $body['error'] : 'HTTP ' . $code;
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'GotASale: Retry #' . $new_attempt . ' failed — ' . $error );
		}
		self::save_registration_failure( $error, $new_attempt );
		self::schedule_retry( $new_attempt );

		return get_option( 'gotasale_registration_status' );
	}

	/**
	 * Save a failed registration status.
	 *
	 * @param string $error    Error message.
	 * @param int    $attempts Number of attempts so far.
	 */
	private static function save_registration_failure( $error, $attempts ) {
		update_option( 'gotasale_registration_status', array(
			'status'       => 'failed',
			'error'        => $error,
			'attempts'     => $attempts,
			'last_attempt' => time(),
		) );
	}

	/**
	 * Schedule a retry with exponential backoff.
	 *
	 * @param int $attempts Number of attempts completed so far.
	 */
	private static function schedule_retry( $attempts ) {
		if ( $attempts >= self::$max_attempts ) {
			return;
		}

		// Pick delay based on attempt number (clamp to last entry if beyond array)
		$index = min( $attempts - 1, count( self::$backoff_delays ) - 1 );
		$delay = self::$backoff_delays[ max( 0, $index ) ];

		wp_schedule_single_event( time() + $delay, 'gotasale_retry_registration' );
	}
}
