/**
 * GotASale Admin JavaScript
 *
 * Handles dynamic UI: theme toggle, copy token, test notification, upgrade/manage actions,
 * destination management, pending link approvals, per-destination configuration,
 * and registration retry.
 */

(function ($) {
  "use strict";

  var channelsData = [];

  var platformLabels = {
    telegram: "Telegram",
    discord: "Discord",
    slack: "Slack",
    webhook: "Webhook",
    push: "Push",
  };

  $(document).ready(function () {
    // ── Theme Toggle ───────────────────────────────────
    var $settings = $(".gotasale-settings");

    var savedTheme = localStorage.getItem("gotasale-theme") || "dark";
    $settings.attr("data-theme", savedTheme);
    updateThemeIcon(savedTheme);

    $("#gotasale-theme-switch").on("click", function () {
      var current = $settings.attr("data-theme");
      var next = current === "dark" ? "light" : "dark";
      $settings.attr("data-theme", next);
      localStorage.setItem("gotasale-theme", next);
      updateThemeIcon(next);
    });

    function updateThemeIcon(theme) {
      // Moon for dark, Sun for light
      $("#gotasale-theme-icon").html(theme === "dark" ? "&#9790;" : "&#9728;");
    }

    // ── Copy site token to clipboard ───────────────────
    $("#gotasale-copy-token").on("click", function () {
      var token = $("#gotasale-site-token").text();
      navigator.clipboard.writeText(token).then(function () {
        $("#gotasale-copy-token").text("Copied!");
        setTimeout(function () {
          $("#gotasale-copy-token").text("Copy");
        }, 2000);
      });
    });

    // ── Send test notification ─────────────────────────
    $("#gotasale-send-test").on("click", function () {
      var $btn = $(this);
      var $result = $("#gotasale-test-result");

      $btn.prop("disabled", true);
      $result.text("Sending...").css("color", "");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_send_test",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success) {
            $result.text("Test notification sent!").css("color", "#0cf59c");
          } else {
            $result
              .text(response.data || "Failed to send test notification.")
              .css("color", "#f5400c");
          }
        },
        error: function () {
          $result.text("Network error.").css("color", "#f5400c");
        },
        complete: function () {
          $btn.prop("disabled", false);
        },
      });
    });

    // ── Upgrade checkout buttons (monthly/yearly) ─────
    $(document).on("click", ".gotasale-checkout-btn", function (e) {
      e.preventDefault();
      var $btn = $(this);
      var tier = $btn.data("tier");
      var interval = $btn.data("interval") || "month";
      $btn.data("original-text", $btn.text());
      $btn.text("Loading...").prop("disabled", true).css("opacity", "0.6");
      gotasaleCreateCheckout(tier, interval, $btn);
    });


    // ── Retry Registration ─────────────────────────────
    $("#gotasale-retry-registration").on("click", function () {
      var $btn = $(this);
      $btn.prop("disabled", true).text("Retrying...");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_retry_registration",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success && response.data && response.data.status === "registered") {
            // Replace warning banner with success message
            $("#gotasale-registration-warning").html(
              '<div class="gotasale-registration-warning__content">' +
                '<span class="gotasale-registration-warning__icon">&#10003;</span>' +
                '<div class="gotasale-registration-warning__text" style="color: #0cf59c;">' +
                  "Store registered successfully! Reload the page to dismiss this banner." +
                "</div>" +
              "</div>"
            ).css({
              "border-color": "rgba(12, 245, 156, 0.3)",
              "background": "rgba(12, 245, 156, 0.08)"
            });
          } else {
            var error = (response.data && response.data.error) || "Registration failed. Will retry automatically.";
            $btn.prop("disabled", false).text("Retry Now");
            $btn.before(
              '<span class="gotasale-text-error" style="margin-right:10px;font-size:13px;">' +
                escapeHtml(error) +
              "</span>"
            );
          }
        },
        error: function () {
          $btn.prop("disabled", false).text("Retry Now");
        },
      });
    });

    // ── Unlink (Deactivate) License Key ──────────────
    $("#gotasale-unlink-license").on("click", function () {
      if (
        !confirm(
          "Are you sure? This will unlink this store from your account and revert it to the free tier."
        )
      ) {
        return;
      }

      var $btn = $(this);
      var $result = $("#gotasale-license-result");

      $btn.prop("disabled", true);
      $result.text("Unlinking...").css("color", "");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_deactivate_license",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success) {
            $result
              .text("License unlinked! Reloading...")
              .css("color", "#0cf59c");
            setTimeout(function () {
              location.reload();
            }, 1000);
          } else {
            $result
              .text(response.data || "Unlink failed.")
              .css("color", "#f5400c");
            $btn.prop("disabled", false);
          }
        },
        error: function () {
          $result.text("Network error.").css("color", "#f5400c");
          $btn.prop("disabled", false);
        },
      });
    });

    // ── Activate License Key ─────────────────────────
    $("#gotasale-activate-license").on("click", function () {
      var $btn = $(this);
      var $result = $("#gotasale-license-result");
      var licenseKey = $("#gotasale-license-key").val().trim();

      if (!licenseKey) {
        $result.text("Please enter a license key.").css("color", "#f5400c");
        return;
      }

      $btn.prop("disabled", true);
      $result.text("Activating...").css("color", "");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_activate_license",
          nonce: gotasaleAdmin.nonce,
          license_key: licenseKey,
        },
        success: function (response) {
          if (response.success) {
            $result.text("License activated! Reloading...").css("color", "#0cf59c");
            setTimeout(function () {
              location.reload();
            }, 1000);
          } else {
            $result
              .text(response.data || "Activation failed.")
              .css("color", "#f5400c");
            $btn.prop("disabled", false);
          }
        },
        error: function () {
          $result.text("Network error.").css("color", "#f5400c");
          $btn.prop("disabled", false);
        },
      });
    });

    // ── Refresh pending links manually ────────────────
    $("#gotasale-refresh-pending").on("click", function () {
      var $btn = $(this);
      $btn.prop("disabled", true).css("opacity", "0.5");
      loadPendingLinks();
      setTimeout(function () {
        $btn.prop("disabled", false).css("opacity", "");
      }, 1000);
    });

    // ── Copy connect commands to clipboard ────────────
    $(".gotasale-copy-command").on("click", function () {
      var $btn = $(this);
      var command = $btn.closest(".gotasale-connect-command").find("code").text();
      navigator.clipboard.writeText(command).then(function () {
        $btn.text("Copied!");
        setTimeout(function () {
          $btn.text("Copy");
        }, 2000);
      });
    });

    // ── Load pending links and destinations on page load ──
    loadPendingLinks();
    loadDestinations();

    // Poll for new pending links every 15 seconds
    setInterval(loadPendingLinks, 15000);

    /**
     * Create a Stripe checkout session and redirect.
     */
    function gotasaleCreateCheckout(tier, interval, $btn) {
      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_create_checkout",
          nonce: gotasaleAdmin.nonce,
          tier: tier,
          interval: interval || "month",
        },
        success: function (response) {
          if (response.success && response.data.url) {
            window.location.href = response.data.url;
          } else if ($btn) {
            $btn
              .text($btn.data("original-text"))
              .prop("disabled", false)
              .css("opacity", "");
          }
        },
        error: function () {
          if ($btn) {
            $btn
              .text($btn.data("original-text"))
              .prop("disabled", false)
              .css("opacity", "");
          }
        },
      });
    }

    /**
     * Load and render pending link requests.
     */
    function loadPendingLinks() {
      var $list = $("#gotasale-pending-list");
      var $empty = $("#gotasale-pending-empty");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_get_pending_links",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (
            response.success &&
            response.data &&
            response.data.pending_links &&
            response.data.pending_links.length > 0
          ) {
            renderPendingLinks(response.data.pending_links);
          } else {
            $list.empty();
            $empty.show();
          }
        },
        error: function () {
          $list.html(
            '<div class="gotasale-loading gotasale-text-error">Failed to load pending links.</div>',
          );
        },
      });
    }

    /**
     * Render pending link requests.
     */
    function renderPendingLinks(links) {
      var $list = $("#gotasale-pending-list");
      var $empty = $("#gotasale-pending-empty");

      $list.empty();
      $empty.hide();

      if (!links || links.length === 0) {
        $empty.show();
        return;
      }

      $.each(links, function (i, link) {
        var title = link.chat_title || link.chat_id || "\u2014";
        var expiresIn = getTimeRemaining(link.expires_at);

        var $row = $(
          '<div class="gotasale-pending-row">' +
            '<div class="gotasale-pending-info">' +
            '<span class="gotasale-platform-badge gotasale-platform-badge--' +
            link.platform +
            '">' +
            platformLabels[link.platform] +
            "</span>" +
            '<span class="gotasale-pending-title">' +
            escapeHtml(title) +
            "</span>" +
            '<span class="gotasale-pending-code">Code: <strong>' +
            escapeHtml(link.confirm_code) +
            "</strong></span>" +
            '<span class="gotasale-pending-expires">Expires: ' +
            expiresIn +
            "</span>" +
            "</div>" +
            '<div class="gotasale-pending-actions">' +
            '<button type="button" class="gotasale-btn gotasale-btn--primary gotasale-btn--sm gotasale-approve-btn" data-code="' +
            escapeHtml(link.confirm_code) +
            '">Approve</button>' +
            "</div>" +
            "</div>",
        );

        $list.append($row);
      });

      // Approve button click
      $list.find(".gotasale-approve-btn").on("click", function () {
        var code = $(this).data("code");
        confirmLink(code);
      });
    }

    /**
     * Confirm a pending link request.
     */
    function confirmLink(code) {
      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_confirm_link",
          nonce: gotasaleAdmin.nonce,
          confirm_code: code,
        },
        success: function (response) {
          if (response.success) {
            // Refresh both lists
            loadPendingLinks();
            loadDestinations();
          } else {
            alert(response.data || "Failed to confirm link.");
          }
        },
        error: function () {
          alert("Network error. Please try again.");
        },
      });
    }

    /**
     * Calculate time remaining from an ISO date string.
     */
    function getTimeRemaining(expiresAt) {
      var now = new Date();
      var expires = new Date(expiresAt);
      var diff = expires - now;

      if (diff <= 0) return "Expired";

      var minutes = Math.ceil(diff / 60000);
      return minutes + " min";
    }

    /**
     * Load and render the destinations list.
     * On initial load, uses data embedded by PHP via wp_localize_script.
     * On subsequent calls (after mutations), fetches fresh data via AJAX.
     */
    function loadDestinations() {
      var $list = $("#gotasale-channel-list");
      var $empty = $("#gotasale-empty-state");

      // First load: use data from PHP (avoids extra round-trip)
      if (gotasaleAdmin.channelsData && !loadDestinations._loaded) {
        loadDestinations._loaded = true;
        channelsData = gotasaleAdmin.channelsData.channels || [];
        renderDestinations();
        return;
      }

      // Subsequent loads: fetch fresh via AJAX
      $list.html(
        '<div class="gotasale-loading">Loading destinations...</div>',
      );
      $empty.hide();

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_get_channels",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success && response.data && response.data.channels) {
            channelsData = response.data.channels;
          } else {
            channelsData = [];
          }
          renderDestinations();
        },
        error: function () {
          $list.html(
            '<div class="gotasale-loading gotasale-text-error">Failed to load destinations.</div>',
          );
        },
      });
    }

    /**
     * Render the channel list from channelsData.
     */
    function renderDestinations() {
      var $list = $("#gotasale-channel-list");
      var $empty = $("#gotasale-empty-state");

      $list.empty();

      if (!channelsData || channelsData.length === 0) {
        $empty.show();
        return;
      }

      $empty.hide();

      $.each(channelsData, function (i, channel) {
        var eventsLabel = summarizePreferences(channel.preferences);
        var fieldsLabel = summarizeFields(channel.payload_config);
        var title = channel.chat_title || channel.chat_id || "\u2014";

        // Truncate long webhook URLs for display
        if (channel.platform === "webhook" && title.length > 40) {
          title = title.substring(0, 37) + "...";
        }

        var $row = $(
          '<div class="gotasale-channel-row" data-channel-id="' +
            channel.id +
            '">' +
            '<div class="gotasale-channel-info">' +
            '<span class="gotasale-platform-badge gotasale-platform-badge--' +
            channel.platform +
            '">' +
            platformLabels[channel.platform] +
            "</span>" +
            '<span class="gotasale-channel-title">' +
            escapeHtml(title) +
            "</span>" +
            "</div>" +
            '<div class="gotasale-channel-meta">' +
            '<span class="gotasale-channel-events">Events: ' +
            eventsLabel +
            "</span>" +
            '<span class="gotasale-channel-fields">Fields: ' +
            fieldsLabel +
            "</span>" +
            "</div>" +
            '<div class="gotasale-channel-actions">' +
            '<button type="button" class="gotasale-btn gotasale-btn--danger gotasale-btn--sm gotasale-remove-btn" data-id="' +
            channel.id +
            '">Remove</button>' +
            "</div>" +
            "</div>",
        );

        $list.append($row);
      });

      // Remove button
      $list.find(".gotasale-remove-btn").on("click", function (e) {
        e.stopPropagation();
        var id = $(this).data("id");
        removeChannel(id);
      });
    }

    /**
     * Summarize preferences into a display label.
     */
    function summarizePreferences(prefs) {
      if (!prefs || Object.keys(prefs).length === 0) return "All";
      var enabled = [];
      if (prefs.new_order !== false) enabled.push("New Order");
      if (prefs.status_changed !== false) enabled.push("Status Changed");
      if (prefs.low_stock !== false) enabled.push("Low Stock");
      if (prefs.refund_created !== false) enabled.push("Refund");
      if (prefs.payment_failed !== false) enabled.push("Failed");
      if (prefs.order_cancelled !== false) enabled.push("Cancelled");
      if (enabled.length === 6) return "All";
      if (enabled.length === 0) return "None";
      return enabled.join(", ");
    }

    /**
     * Summarize payload config fields into a display label.
     */
    function summarizeFields(config) {
      if (!config || !config.fields) return "All";
      var labels = {
        order_info: "Order Info",
        customer_name: "Customer Name",
        customer_contact: "Customer Contact",
        items: "Items",
        pricing: "Pricing",
        address: "Address",
      };
      if (config.fields.length >= 6) return "All";
      return config.fields.map(function (f) { return labels[f] || f; }).join(", ");
    }

    /**
     * Remove a channel destination.
     */
    function removeChannel(id) {
      if (!confirm("Remove this destination?")) return;

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_remove_channel",
          nonce: gotasaleAdmin.nonce,
          channel_id: id,
        },
        success: function (response) {
          if (response.success) {
            // Remove from local data and re-render
            channelsData = channelsData.filter(function (ch) {
              return ch.id !== id;
            });
            renderDestinations();
          }
        },
      });
    }

    // ── Reconnect WC API ──────────────────────────────────
    $("#gotasale-reconnect-wc-api").on("click", function () {
      var $btn = $(this);
      $btn.prop("disabled", true).text("Reconnecting...");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_reconnect_wc_api",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success) {
            $btn.text("Connected!").css("background", "rgba(12,245,156,0.2)");
            setTimeout(function () {
              location.reload();
            }, 1000);
          } else {
            $btn.text("Retry").prop("disabled", false);
          }
        },
        error: function () {
          $btn.text("Retry").prop("disabled", false);
        },
      });
    });

    // ── Upgrade Connection (read → read_write) ─────────────
    $("#gotasale-upgrade-connection").on("click", function () {
      var $btn = $(this);
      $btn.prop("disabled", true).text("Upgrading...");

      $.ajax({
        url: gotasaleAdmin.ajaxUrl,
        method: "POST",
        data: {
          action: "gotasale_reconnect_wc_api",
          nonce: gotasaleAdmin.nonce,
        },
        success: function (response) {
          if (response.success) {
            $btn.text("Upgraded!").css("background", "rgba(12,245,156,0.2)");
            setTimeout(function () {
              location.reload();
            }, 1000);
          } else {
            $btn.text("Retry").prop("disabled", false);
          }
        },
        error: function () {
          $btn.text("Retry").prop("disabled", false);
        },
      });
    });

    /**
     * Escape HTML to prevent XSS.
     */
    function escapeHtml(str) {
      var div = document.createElement("div");
      div.appendChild(document.createTextNode(str));
      return div.innerHTML;
    }
  });
})(jQuery);
