<?php
/**
 * Admin page registration and rendering.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class GotASale_Admin {

	/**
	 * Register admin hooks.
	 */
	public function init() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_filter( 'plugin_action_links_' . plugin_basename( GOTASALE_PLUGIN_DIR . 'gotasale.php' ), array( $this, 'add_settings_link' ) );

		// AJAX handlers for admin UI actions.
		add_action( 'wp_ajax_gotasale_send_test', array( $this, 'ajax_send_test' ) );
		add_action( 'wp_ajax_gotasale_create_checkout', array( $this, 'ajax_create_checkout' ) );
		add_action( 'wp_ajax_gotasale_billing_portal', array( $this, 'ajax_billing_portal' ) );
		add_action( 'wp_ajax_gotasale_get_channels', array( $this, 'ajax_get_channels' ) );
		add_action( 'wp_ajax_gotasale_remove_channel', array( $this, 'ajax_remove_channel' ) );
		add_action( 'wp_ajax_gotasale_get_pending_links', array( $this, 'ajax_get_pending_links' ) );
		add_action( 'wp_ajax_gotasale_confirm_link', array( $this, 'ajax_confirm_link' ) );
		add_action( 'wp_ajax_gotasale_retry_registration', array( $this, 'ajax_retry_registration' ) );
		add_action( 'wp_ajax_gotasale_reconnect_wc_api', array( $this, 'ajax_reconnect_wc_api' ) );
		add_action( 'wp_ajax_gotasale_activate_license', array( $this, 'ajax_activate_license' ) );
		add_action( 'wp_ajax_gotasale_deactivate_license', array( $this, 'ajax_deactivate_license' ) );
	}

	/**
	 * Add the settings page to the WordPress admin menu.
	 */
	public function add_menu_page() {
		add_options_page(
			__( 'Got A Sale Settings', 'got-a-sale' ),
			__( 'Got A Sale', 'got-a-sale' ),
			'manage_woocommerce',
			'got-a-sale',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Add Settings link on the Plugins page.
	 *
	 * @param array $links Existing action links.
	 * @return array Modified action links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( 'options-general.php?page=got-a-sale' ) ),
			esc_html__( 'Settings', 'got-a-sale' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Enqueue admin CSS and JS.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_assets( $hook ) {
		if ( 'settings_page_got-a-sale' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'gotasale-admin',
			GOTASALE_PLUGIN_URL . 'admin/css/gotasale-admin.css',
			array(),
			GOTASALE_VERSION
		);

		wp_enqueue_script(
			'gotasale-admin',
			GOTASALE_PLUGIN_URL . 'admin/js/gotasale-admin.js',
			array( 'jquery' ),
			GOTASALE_VERSION,
			true
		);

		$plan_status        = get_option( 'gotasale_plan_status', array( 'tier' => 'free' ) );
		$channels_data      = GotASale_API::get_available_channels();
		$registration_status = get_option( 'gotasale_registration_status', array() );

		wp_localize_script( 'gotasale-admin', 'gotasaleAdmin', array(
			'ajaxUrl'            => admin_url( 'admin-ajax.php' ),
			'nonce'              => wp_create_nonce( 'gotasale_admin' ),
			'tier'               => isset( $plan_status['tier'] ) ? $plan_status['tier'] : 'free',
			'channelsData'       => is_wp_error( $channels_data ) ? null : $channels_data,
			'registrationStatus' => $registration_status,
		) );
	}

	/**
	 * Render the settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		include GOTASALE_PLUGIN_DIR . 'admin/views/settings-page.php';
	}

	/**
	 * AJAX: Send a test notification.
	 */
	public function ajax_send_test() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_API::send_test();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Create a Stripe checkout session and return the URL.
	 */
	public function ajax_create_checkout() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$tier     = isset( $_POST['tier'] ) ? sanitize_text_field( wp_unslash( $_POST['tier'] ) ) : '';
		$interval = isset( $_POST['interval'] ) ? sanitize_text_field( wp_unslash( $_POST['interval'] ) ) : 'month';

		if ( ! in_array( $tier, array( 'pro', 'agency' ), true ) ) {
			wp_send_json_error( __( 'Invalid tier.', 'got-a-sale' ) );
		}

		if ( ! in_array( $interval, array( 'month', 'year' ), true ) ) {
			$interval = 'month';
		}

		$result = GotASale_API::create_checkout_session( $tier, $interval );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		// Validate redirect URL hostname
		if ( isset( $result['url'] ) && ! self::validate_stripe_url( $result['url'] ) ) {
			wp_send_json_error( __( 'Invalid checkout URL received.', 'got-a-sale' ) );
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Create a Stripe billing portal session and return the URL.
	 */
	public function ajax_billing_portal() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_API::create_billing_portal();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		// Validate redirect URL hostname
		if ( isset( $result['url'] ) && ! self::validate_stripe_url( $result['url'] ) ) {
			wp_send_json_error( __( 'Invalid portal URL received.', 'got-a-sale' ) );
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Get fresh channel data (bypasses cache).
	 */
	public function ajax_get_channels() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_API::get_available_channels();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Remove a channel destination.
	 */
	public function ajax_remove_channel() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$id = isset( $_POST['channel_id'] ) ? absint( $_POST['channel_id'] ) : 0;

		if ( ! $id ) {
			wp_send_json_error( __( 'Invalid channel ID.', 'got-a-sale' ) );
		}

		$result = GotASale_API::remove_channel( $id );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( $result );
	}


	/**
	 * AJAX: Get pending link requests.
	 */
	public function ajax_get_pending_links() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_API::get_pending_links();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		// Auto-heal: server reports credentials missing → reconnect
		if ( isset( $result['wc_status'] ) && 'missing' === $result['wc_status'] && GotASale_WC_API::is_configured() ) {
			delete_option( 'gotasale_wc_api_configured' );
			delete_option( 'gotasale_wc_permissions' );
			GotASale_WC_API::setup();
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Confirm a pending link request.
	 */
	public function ajax_confirm_link() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$code = isset( $_POST['confirm_code'] ) ? sanitize_text_field( wp_unslash( $_POST['confirm_code'] ) ) : '';

		if ( empty( $code ) ) {
			wp_send_json_error( __( 'Confirmation code is required.', 'got-a-sale' ) );
		}

		$result = GotASale_API::confirm_link( $code );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Retry store registration.
	 */
	public function ajax_retry_registration() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$status = GotASale_Activator::retry_registration();
		wp_send_json_success( $status );
	}

	/**
	 * AJAX: Reconnect WooCommerce API keys.
	 */
	public function ajax_reconnect_wc_api() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_WC_API::reconnect();
		if ( $result ) {
			wp_send_json_success( array( 'status' => 'connected' ) );
		} else {
			wp_send_json_error( __( 'Failed to reconnect. Will retry automatically.', 'got-a-sale' ) );
		}
	}

	/**
	 * AJAX: Deactivate (unlink) a license key.
	 */
	public function ajax_deactivate_license() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$result = GotASale_API::deactivate_license();

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		delete_option( 'gotasale_license_key' );
		delete_option( 'gotasale_plan_status' );

		wp_send_json_success( $result );
	}

	/**
	 * AJAX: Activate a license key.
	 */
	public function ajax_activate_license() {
		check_ajax_referer( 'gotasale_admin', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( __( 'Permission denied.', 'got-a-sale' ), 403 );
		}

		$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( __( 'License key is required.', 'got-a-sale' ) );
		}

		$result = GotASale_API::activate_license( $license_key );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}

		update_option( 'gotasale_license_key', $license_key );

		if ( ! empty( $result['tier'] ) ) {
			update_option( 'gotasale_plan_status', array( 'tier' => $result['tier'] ) );
		}

		wp_send_json_success( $result );
	}


	/**
	 * Validates the redirect URL hostname before returning.
	 */
	private static function validate_stripe_url( $url ) {
		if ( empty( $url ) ) {
			return false;
		}

		$parsed = wp_parse_url( $url );

		if ( empty( $parsed['host'] ) ) {
			return false;
		}

		$allowed_hosts = array(
			'checkout.stripe.com',
			'billing.stripe.com',
		);

		return in_array( $parsed['host'], $allowed_hosts, true );
	}
}
