<?php
/**
 * GotASale — Instant Order Notifications for PrestaShop
 *
 * Sends real-time order notifications to Telegram, Discord, Teams,
 * Google Chat, Slack, and more via the GotASale service.
 *
 * @author    GotASale
 * @copyright 2024-2026 GotASale
 * @license   MIT
 * @version   1.0.0
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once dirname(__FILE__) . '/classes/GotASaleAPI.php';
require_once dirname(__FILE__) . '/classes/GotASaleWebservice.php';

class GotASale extends Module
{
    public function __construct()
    {
        $this->name = 'gotasale';
        $this->tab = 'administration';
        $this->version = '1.0.0';
        $this->author = 'GotASale';
        $this->need_instance = 0;
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('GotASale — Instant Order Notifications');
        $this->description = $this->l('Get instant order notifications on Telegram, Discord, Teams, Google Chat, or Slack.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall GotASale?');
        $this->ps_versions_compliancy = ['min' => '1.7.7.0', 'max' => _PS_VERSION_];
    }

    /**
     * Install the module: register hooks, generate site token, register with server.
     */
    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        // Register hooks
        if (!$this->registerHook('actionValidateOrderAfter')
            || !$this->registerHook('actionOrderStatusPostUpdate')
        ) {
            return false;
        }

        // Generate site token
        $siteToken = $this->generateSiteToken();
        Configuration::updateValue('GOTASALE_SITE_TOKEN', $siteToken);
        Configuration::updateValue('GOTASALE_API_URL', 'https://api.gotasale.io');

        // Register store with GotASale server
        $api = new GotASaleAPI();
        $api->registerStore($siteToken, Tools::getShopDomainSsl(true), Configuration::get('PS_SHOP_NAME'));

        // Create Webservice API key and send to server
        $webservice = new GotASaleWebservice();
        $webservice->createAndSendApiKey($siteToken);

        return true;
    }

    /**
     * Uninstall the module.
     */
    public function uninstall()
    {
        Configuration::deleteByName('GOTASALE_SITE_TOKEN');
        Configuration::deleteByName('GOTASALE_API_URL');
        Configuration::deleteByName('GOTASALE_WS_KEY_ID');

        // Remove the Webservice key
        $webservice = new GotASaleWebservice();
        $webservice->removeApiKey();

        return parent::uninstall();
    }

    /**
     * Module configuration page.
     */
    public function getContent()
    {
        $output = '';

        // Handle confirm link submission
        if (Tools::isSubmit('submitGotaSaleConfirm')) {
            $confirmCode = Tools::getValue('confirm_code');
            $siteToken = Configuration::get('GOTASALE_SITE_TOKEN');
            $api = new GotASaleAPI();
            $result = $api->confirmLink($siteToken, $confirmCode);
            if ($result && isset($result['ok']) && $result['ok']) {
                $output .= $this->displayConfirmation($this->l('Channel linked successfully!'));
            } else {
                $output .= $this->displayError($this->l('Failed to confirm link. It may have expired.'));
            }
        }

        // Handle test notification
        if (Tools::isSubmit('submitGotaSaleSendTest')) {
            $siteToken = Configuration::get('GOTASALE_SITE_TOKEN');
            $api = new GotASaleAPI();
            $result = $api->sendTest($siteToken);
            if ($result && isset($result['ok']) && $result['ok']) {
                $output .= $this->displayConfirmation($this->l('Test notification sent!'));
            } else {
                $output .= $this->displayError($this->l('Failed to send test notification. Make sure you have at least one linked channel.'));
            }
        }

        // Handle settings submission
        if (Tools::isSubmit('submitGotaSaleSettings')) {
            $apiUrl = Tools::getValue('GOTASALE_API_URL');
            if ($apiUrl && Validate::isUrl($apiUrl)) {
                Configuration::updateValue('GOTASALE_API_URL', rtrim($apiUrl, '/'));
                $output .= $this->displayConfirmation($this->l('Settings saved.'));
            }
        }

        // Get pending links
        $siteToken = Configuration::get('GOTASALE_SITE_TOKEN');
        $api = new GotASaleAPI();
        $pendingLinks = $api->getPendingLinks($siteToken);

        $this->context->smarty->assign([
            'site_token' => $siteToken,
            'api_url' => Configuration::get('GOTASALE_API_URL'),
            'pending_links' => $pendingLinks,
            'module_dir' => $this->_path,
            'confirm_action' => $this->context->link->getAdminLink('AdminModules', true, [], [
                'configure' => $this->name,
                'tab_module' => $this->tab,
                'module_name' => $this->name,
            ]),
        ]);

        return $output . $this->display(__FILE__, 'views/templates/admin/configure.tpl');
    }

    // ── Hook Handlers ────────────────────────────────────────

    /**
     * Hook: After order is fully created.
     * Sends a "new_order" notification to GotASale server.
     */
    public function hookActionValidateOrderAfter($params)
    {
        if (!isset($params['order']) || !($params['order'] instanceof Order)) {
            return;
        }

        $order = $params['order'];
        $this->sendNotification('new_order', $order);
    }

    /**
     * Hook: After order status is changed.
     * Sends a "status_changed" notification to GotASale server.
     */
    public function hookActionOrderStatusPostUpdate($params)
    {
        if (!isset($params['id_order'])) {
            return;
        }

        $order = new Order((int) $params['id_order']);
        if (!Validate::isLoadedObject($order)) {
            return;
        }

        $newStatus = isset($params['newOrderStatus']) ? $params['newOrderStatus']->name : '';
        $oldStatus = '';
        if (isset($params['oldOrderStatus'])) {
            $oldStatus = $params['oldOrderStatus']->name;
        }

        $this->sendNotification('status_changed', $order, [
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
        ]);
    }

    // ── Notification Builder ─────────────────────────────────

    /**
     * Build and send a notification payload to the GotASale server.
     */
    private function sendNotification($event, Order $order, array $extra = [])
    {
        $siteToken = Configuration::get('GOTASALE_SITE_TOKEN');
        if (!$siteToken) {
            return;
        }

        $customer = new Customer($order->id_customer);
        $currency = new Currency($order->id_currency);
        $invoiceAddress = new Address($order->id_address_invoice);
        $deliveryAddress = new Address($order->id_address_delivery);

        // Build order items
        $items = [];
        $products = $order->getProductsDetail();
        foreach ($products as $product) {
            $items[] = [
                'name' => $product['product_name'],
                'quantity' => (int) $product['product_quantity'],
                'total' => number_format((float) $product['total_price_tax_incl'], 2, '.', ''),
                'sku' => $product['product_reference'] ?? '',
            ];
        }

        // Build payload
        $payload = [
            'event' => $event,
            'order' => [
                'id' => (int) $order->id,
                'number' => $order->reference,
                'status' => $this->getOrderStateName($order->current_state),
                'total' => number_format((float) $order->total_paid_tax_incl, 2, '.', ''),
                'currency' => $currency->iso_code,
                'payment_method' => $order->payment,
                'date_created' => $order->date_add,
            ],
            'customer' => [
                'name' => trim($customer->firstname . ' ' . $customer->lastname),
                'email' => $customer->email,
                'phone' => $invoiceAddress->phone ?: $invoiceAddress->phone_mobile,
            ],
            'items' => $items,
            'pricing' => [
                'subtotal' => number_format((float) $order->total_products_wt, 2, '.', ''),
                'shipping' => number_format((float) $order->total_shipping_tax_incl, 2, '.', ''),
                'tax' => number_format((float) $order->total_paid_tax_incl - (float) $order->total_paid_tax_excl, 2, '.', ''),
                'total' => number_format((float) $order->total_paid_tax_incl, 2, '.', ''),
                'payment_method' => $order->payment,
            ],
            'address' => [
                'billing_country' => Country::getIsoById($invoiceAddress->id_country),
                'billing_state' => State::getNameById($invoiceAddress->id_state),
                'billing_city' => $invoiceAddress->city,
                'shipping_country' => Country::getIsoById($deliveryAddress->id_country),
                'shipping_state' => State::getNameById($deliveryAddress->id_state),
                'shipping_city' => $deliveryAddress->city,
            ],
        ];

        // Add store name
        $payload['storeName'] = Configuration::get('PS_SHOP_NAME') ?: Tools::getShopDomainSsl(true);

        // Merge extra data (old_status, new_status, etc.)
        $payload = array_merge($payload, $extra);

        $api = new GotASaleAPI();
        $api->sendNotification($siteToken, $payload);
    }

    /**
     * Get human-readable order state name.
     */
    private function getOrderStateName($stateId)
    {
        $state = new OrderState((int) $stateId, $this->context->language->id);
        return Validate::isLoadedObject($state) ? $state->name : 'Unknown';
    }

    /**
     * Generate a unique site token.
     */
    private function generateSiteToken()
    {
        return 'ps_' . bin2hex(random_bytes(16));
    }
}
