<?php
/**
 * GotASale Webservice API Key Manager for PrestaShop.
 * Creates a read-only Webservice API key for order proxying
 * and sends it to the GotASale server for encrypted storage.
 */

class GotASaleWebservice
{
    /**
     * Create a Webservice API key and send it to the GotASale server.
     */
    public function createAndSendApiKey($siteToken)
    {
        // Check if we already have a key
        $existingKeyId = Configuration::get('GOTASALE_WS_KEY_ID');
        if ($existingKeyId) {
            $existingKey = new WebserviceKey((int) $existingKeyId);
            if (Validate::isLoadedObject($existingKey)) {
                // Key already exists, just resend it
                $api = new GotASaleAPI();
                $api->sendApiKey($siteToken, $existingKey->key);
                return;
            }
        }

        // Generate a new API key
        $apiKey = $this->generateApiKey();

        // Create the WebserviceKey object
        $wsKey = new WebserviceKey();
        $wsKey->key = $apiKey;
        $wsKey->description = 'GotASale - Order Notifications (read-only)';
        $wsKey->active = 1;

        if (!$wsKey->add()) {
            PrestaShopLogger::addLog(
                'GotASale: Failed to create Webservice API key',
                3,
                null,
                'GotASale',
                null,
                true
            );
            return;
        }

        // Set permissions: read-only for orders, customers, addresses, currencies
        $resources = [
            'orders' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'customers' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'addresses' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'currencies' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'order_states' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'countries' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
            'states' => ['GET' => 1, 'POST' => 0, 'PUT' => 0, 'DELETE' => 0, 'HEAD' => 1],
        ];

        foreach ($resources as $resource => $methods) {
            foreach ($methods as $method => $access) {
                if ($access) {
                    Db::getInstance()->insert('webservice_permission', [
                        'id_webservice_account' => (int) $wsKey->id,
                        'resource' => pSQL($resource),
                        'method' => pSQL($method),
                    ]);
                }
            }
        }

        // Save the key ID for later cleanup
        Configuration::updateValue('GOTASALE_WS_KEY_ID', $wsKey->id);

        // Enable Webservice if not already enabled
        if (!Configuration::get('PS_WEBSERVICE')) {
            Configuration::updateValue('PS_WEBSERVICE', 1);
        }

        // Send the key to GotASale server
        $api = new GotASaleAPI();
        $api->sendApiKey($siteToken, $apiKey);
    }

    /**
     * Remove the Webservice API key on uninstall.
     */
    public function removeApiKey()
    {
        $keyId = Configuration::get('GOTASALE_WS_KEY_ID');
        if (!$keyId) {
            return;
        }

        $wsKey = new WebserviceKey((int) $keyId);
        if (Validate::isLoadedObject($wsKey)) {
            // Remove permissions first
            Db::getInstance()->delete('webservice_permission', 'id_webservice_account = ' . (int) $wsKey->id);
            $wsKey->delete();
        }
    }

    /**
     * Generate a random 32-character API key.
     */
    private function generateApiKey()
    {
        return bin2hex(random_bytes(16));
    }
}
